<?php

namespace Paymill\Models\Response;

/**
 * Transaction Model
 * A transaction is the charging of a credit card or a direct debit.
 * In this case you need a new transaction object with either a valid token, payment, client + payment or
 * preauthorization. Every transaction has a unique identifier which will be generated by Paymill to identify every
 * transaction. You can issue/create, list and display transactions in detail. Refunds can be done in an extra entity.
 * @tutorial https://paymill.com/de-de/dokumentation/referenz/api-referenz/#document-transactions
 */
class Transaction extends Base
{
    /**
     * @todo REFACTOR: define all response fields and replace usage of hardcoded strings in other classes
     */
    const RESPONSE_FIELD_SHIPPING_ADDRESS = 'shipping_address';
    const RESPONSE_FIELD_BILLING_ADDRESS  = 'billing_address';
    const RESPONSE_FIELD_ITEMS            = 'items';

    /**
     * 'real' amount
     *
     * @var string
     */
    private $_amount;

    /**
     * origin amount
     *
     * @var integer
     */
    private $_originAmount;

    /**
     * Status
     * Possible status values (open, closed, failed, preauth, pending, refunded, partially_refunded, chargeback)
     *
     * @var string
     */
    private $_status;

    /**
     * Description
     *
     * @var string
     */
    private $_description;

    /**
     * Live mode
     *
     * @var boolean
     */
    private $_livemode;

    /**
     * Currency
     *
     * @var string
     */
    private $_currency;

    /**
     * Refunds
     *
     * @var array
     */
    private $_refunds = null;

    /**
     * Response code for transaction feedback. 20000 marks a successful transaction
     * @tutorial https://paymill.com/de-de/dokumentation/referenz/api-referenz/#document-statuscodes
     *
     * @var integer
     */
    private $_responseCode;

    /**
     * Unique identifier of this transaction provided to the acquirer for the statements.
     *
     * @var string
     */
    private $_shortId;

    /**
     * PAYMILL invoice where the transaction fees are charged or null.
     *
     * @var array
     */
    private $_invoices = null;

    /**
     * Payment
     *
     * @var Payment
     */
    private $_payment;

    /**
     * Client
     *
     * @var Client
     */
    private $_client = null;

    /**
     * Preauthorization
     *
     * @var Preauthorization
     */
    private $_preauthorization = null;

    /**
     * Fees
     *
     * @var array
     */
    private $_fees;

    /**
     * Source
     *
     * @var $_source
     */
    private $_source;

    /**
     * Shipping address
     *
     * @var array|null $_shippingAddress
     */
    private $_shippingAddress;

    /**
     * Billing address
     *
     * @var array|null $_billingAddress
     */
    private $_billingAddress;

    /**
     * Items
     *
     * @var array $_items
     */
    private $_items;

    /**
     * Shipping amount
     *
     * @var int $_shippingAmount
     */
    private $_shippingAmount;

    /**
     * Handling amount
     *
     * @var int $_handlingAmount
     */
    private $_handlingAmount;

    /**
     * Returns the 'real' amount.
     *
     * @return string
     */
    public function getAmount()
    {
        return $this->_amount;
    }

    /**
     * Sets the 'real' amount for the transaction.
     * The number must be in the smallest currency unit and will be saved as a string.
     *
     * @param string $amount Amount
     *
     * @return $this
     */
    public function setAmount($amount)
    {
        $this->_amount = $amount;

        return $this;
    }

    /**
     * @var string
     */
    private $_mandateReference;

    /**
     * Returns the origin amount for the transaction.
     *
     * @return integer
     */
    public function getOriginAmount()
    {
        return $this->_originAmount;
    }

    /**
     * Sets the origin amount for the transaction.
     * The number must be in the smallest currency unit and will be saved as a string.
     *
     * @param integer $originAmount Origin amount
     *
     * @return $this
     */
    public function setOriginAmount($originAmount)
    {
        $this->_originAmount = $originAmount;

        return $this;
    }

    /**
     * Returns the transaction status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->_status;
    }

    /**
     * Sets the transaction status.
     *
     * @param string $status Status
     *
     * @return $this
     */
    public function setStatus($status)
    {
        $this->_status = $status;

        return $this;
    }

    /**
     * Returns the transaction description.
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->_description;
    }

    /**
     * Sets the transaction description.
     *
     * @param string $description Description
     *
     * @return $this
     */
    public function setDescription($description)
    {
        $this->_description = $description;

        return $this;
    }

    /**
     * Returns the livemode flag of the transaction.
     *
     * @return boolean
     */
    public function getLivemode()
    {
        return $this->_livemode;
    }

    /**
     * Sets the livemode flag of the transaction.
     *
     * @param boolean $livemode Live mode
     *
     * @return $this
     */
    public function setLivemode($livemode)
    {
        $this->_livemode = $livemode;

        return $this;
    }

    /**
     * Returns the refunds stored in the transaction.
     *
     * @return array|null
     */
    public function getRefunds()
    {
        return $this->_refunds;
    }

    /**
     * Sets the refunds stored in the transaction.
     *
     * @param array $refunds Refunds
     *
     * @return $this
     */
    public function setRefunds($refunds)
    {
        $this->_refunds = $refunds;

        return $this;
    }

    /**
     * Returns the currency.
     *
     * @return string
     */
    public function getCurrency()
    {
        return $this->_currency;
    }

    /**
     * Sets the currency.
     *
     * @param string $currency Currency
     *
     * @return $this
     */
    public function setCurrency($currency)
    {
        $this->_currency = $currency;

        return $this;
    }

    /**
     * Returns the response code (20000 marks a successful transaction)
     *
     * @return integer
     */
    public function getResponseCode()
    {
        return $this->_responseCode;
    }

    /**
     * Sets the response code of the transaction
     *
     * @param integer $responseCode Response code
     *
     * @return $this
     */
    public function setResponseCode($responseCode)
    {
        $this->_responseCode = $responseCode;

        return $this;
    }

    /**
     * Returns the short id of the transaction
     *
     * @return string
     */
    public function getShortId()
    {
        return $this->_shortId;
    }

    /**
     * Sets the transaction short id.
     *
     * @param string $shortId
     *
     * @return $this
     */
    public function setShortId($shortId)
    {
        $this->_shortId = $shortId;

        return $this;
    }

    /**
     * Returns an array of invoices stored in the transaction.
     *
     * @return array|null
     */
    public function getInvoices()
    {
        return $this->_invoices;
    }

    /**
     * Stores an array of invoices in the transaction.
     *
     * @param array $invoices Invoices
     *
     * @return $this
     */
    public function setInvoices($invoices)
    {
        $this->_invoices = $invoices;

        return $this;
    }

    /**
     * Returns the payment associated with the transaction.
     *
     * @return Payment
     */
    public function getPayment()
    {
        return $this->_payment;
    }

    /**
     * Sets the Payment for the transaction.
     *
     * @param Payment $payment Payment
     *
     * @return $this
     */
    public function setPayment($payment)
    {
        $this->_payment = $payment;

        return $this;
    }

    /**
     * Returns the Client associated with the transaction. If no client is available null will be returned.
     *
     * @return Client|null
     */
    public function getClient()
    {
        return $this->_client;
    }

    /**
     * Sets the Client for the transaction
     *
     * @param Client $client Client
     *
     * @return $this
     */
    public function setClient($client)
    {
        $this->_client = $client;

        return $this;
    }

    /**
     * Returns the Preauthorization associated with the transaction. If no preAuth is available null will be returned.
     *
     * @return Preauthorization|null
     */
    public function getPreauthorization()
    {
        return $this->_preauthorization;
    }

    /**
     * Sets the Preauthorization for the transaction.
     *
     * @param Preauthorization $preauthorization Preauthorization
     *
     * @return $this
     */
    public function setPreauthorization($preauthorization)
    {
        $this->_preauthorization = $preauthorization;

        return $this;
    }

    /**
     * Returns the fee array stored in the transaction.
     *
     * @return array
     */
    public function getFees()
    {
        return $this->_fees;
    }


    /**
     * Sets the Fees array for the transaction.
     *
     * @param array $fees Fees
     *
     * @return $this
     */
    public function setFees($fees)
    {
        $this->_fees = $fees;

        return $this;
    }

    /**
     * Sets the name of origin of the call creating the transaction.
     *
     * @param string $source Source
     *
     * @return $this
     */
    public function setSource($source)
    {
        $this->_source = $source;

        return $this;
    }

    /**
     * Gets the name of origin of the call creating the transaction
     *
     * @return string
     */
    public function getSource()
    {
        return $this->_source;
    }

    /**
     * Get shipping address
     *
     * @return array|null
     */
    public function getShippingAddress()
    {
        return $this->_shippingAddress;
    }

    /**
     * Set shipping address
     *
     * @param array|null $shippingAddress Shipping address
     *
     * @return $this
     */
    public function setShippingAddress($shippingAddress)
    {
        $this->_shippingAddress = $shippingAddress;

        return $this;
    }

    /**
     * Get billing address
     *
     * @return array|null
     */
    public function getBillingAddress()
    {
        return $this->_billingAddress;
    }

    /**
     * Set billing address
     *
     * @param array|null $billingAddress Billing address
     *
     * @return $this
     */
    public function setBillingAddress($billingAddress)
    {
        $this->_billingAddress = $billingAddress;

        return $this;
    }

    /**
     * Get items
     *
     * @return array
     */
    public function getItems()
    {
        return $this->_items;
    }

    /**
     * Set items
     *
     * @param array $items Items
     *
     * @return $this
     */
    public function setItems($items)
    {
        $this->_items = $items;

        return $this;
    }

    /**
     * Get shipping amount
     *
     * @return int
     */
    public function getShippingAmount()
    {
        return $this->_shippingAmount;
    }

    /**
     * Set shipping amount
     *
     * @param int $shippingAmount Shipping amount
     *
     * @return $this
     */
    public function setShippingAmount($shippingAmount)
    {
        $this->_shippingAmount = $shippingAmount;

        return $this;
    }

    /**
     * Get handling amount
     *
     * @return int
     */
    public function getHandlingAmount()
    {
        return $this->_handlingAmount;
    }

    /**
     * Set handling amount
     *
     * @param int $handlingAmount Handling amount
     *
     * @return $this
     */
    public function setHandlingAmount($handlingAmount)
    {
        $this->_handlingAmount = $handlingAmount;

        return $this;
    }

    /** Set mandate reference mandate_reference
     * @param string $mandateReference
     */
    public function setMandateReference($mandateReference)
    {
        $this->_mandateReference = $mandateReference;
    }

    /**
     * Return mandate reference mandate_reference
     * @return string
     */
    public function getMandateReference()
    {
        return $this->_mandateReference;
    }
}
